define([
    /* Core */
    'underscore',
    'jquery',
    'backbone',
    'marionette',
    'App',

    /* Resources */
    'modules/new-appointment-request/resources/common/facility/collection',
    'modules/new-appointment-request/resources/direct/clinics/collection',

    /* Views */
    'modules/new-appointment-request/views/common/scheduling-steps/heading-view',
    'modules/new-appointment-request/views/common/facility/section-view',
    'modules/new-appointment-request/views/common/facility/single-item-view',

    /* Templates */
    'text!modules/new-appointment-request/views/common/facility/templates/instructions-no-match.html',
    'text!modules/new-appointment-request/views/common/facility/templates/instructions-one-match.html',
    'text!modules/new-appointment-request/views/common/facility/templates/instructions-default.html',
    'text!modules/new-appointment-request/views/common/scheduling-steps/templates/step-region.html',
], function facilityLayoutView(
    /* Core */
    _,
    $,
    Backbone,
    Mn,
    app,

    /* Resources */
    Collection,
    ClinicsCollection,

    /* Views */
    HeadingView,
    SectionView,
    SingleItemView,

    /* Templates */
    noMatchTemplate,
    oneMatchTemplate,
    defaultMatchTemplate,
    template
) {
    'use strict';

    var LINK = '<a href="https://www.DOMAIN.EX/directory/guide/division.asp?dnum=1" class="external-link">VA Facility Locator</a>';
    
    // eslint-disable-next-line max-len
    var HELP_TEXT = 'If you cannot find the location you need, please visit the ' + LINK + ' to contact the facility for assistance.';
    var EXPRESS_CARE_ID = 'CR1';
    var PRIMARY_CARE_ID = '323';


    /**
     * @class FacilityLayoutView
     * @typedef {Backbone.View<Backbone.Model>}
     */
    var View = Mn.View.extend({
        HELP_TEXT: HELP_TEXT,

        template: _.template(template),

        regions: {
            heading: '.step-heading',
            section: '.step-section',
        },

        initialize: function(options) {
            var typeOfCare = this.model.getCareId();
            var cityState = this.model.get('cityState');
            var facilityCode = cityState.get('rootStationCode');
            var parentCode = cityState.get('institutionCode');

            this.collection = new Collection();
            this.clinicsCollection = new ClinicsCollection();
            this.googleAnalyticsProvider = options.googleAnalyticsProvider;
            this.listenTo(this.collection, 'fetch:success', this.showRegions);

            this.collection.fetchPatientFacilities(typeOfCare, facilityCode, parentCode);
        },

        /**
         * @return {function}
         */
        getInstructionsTemplate: function() {
            var length = this.collection.length;
            if (length === 0) {
                return _.template(noMatchTemplate);
            } else if (length === 1) {
                return _.template(oneMatchTemplate);
            }
            return _.template(defaultMatchTemplate);
        },

        /**
         * Finds the correct instructions based on the number of facilities return for the type of care selected
         * @return {string} html
         */
        getInstructions: function() {
            var instructionTemplate = this.getInstructionsTemplate();
            var data = {
                typeOfCareName: this.options.typeOfCareName,
                helpText: HELP_TEXT,
            };

            return instructionTemplate(data);
        },

        /**
         * @return {void}
         */
        showRegions: function() {
            // eslint-disable-next-line max-len
            if (_.isUndefined(app.disabledFeatures.get('PACT_LOCATION_FEATURE_SET')) && this.model.getCareId() === PRIMARY_CARE_ID) {
                this.collateClinicsForFacilities();
            }
            
            this.showHeading();
            this.showBody();
        },

        /**
         * @return {void}
         */
        showHeading: function() {
            var region = this.getRegion('heading');
            var model = new Backbone.Model({
                title: View.TITLE,
                number: View.STEP_NUMBER,
                instructions: this.getInstructions(),
            });

            region.show(new HeadingView({model: model}));
        },

        /**
         * @return {void}
         */
        showBody: function() {
            var length = this.collection.length;
            var region;
            var view;
            
            if (length === 0) {
                // They won't be able to continue onward no facilities are available
                return;
            }

            region = this.getRegion('section');
            view = length === 1 ? this.getSingleItemView() : this.getSectionView();

            region.show(view);
        },

        /**
         * Gets the express care message.
         * @param {string} typeOfCare
         * @param {Backbone.Model} facility
         * @return {string}
         */
        getExpressMessage: function(typeOfCare, facility) {
            if (typeOfCare === EXPRESS_CARE_ID && facility.isExpressSupportedToday()) {
                return facility.getExpressMessage();
            }
            return '';
        },

        /**
         * Creates a custom View for when there is only one facility available
         * @return {Marionette}
         */
        getSingleItemView: function() {
            var model = this.collection.first();
            var className = model.get('supportsVAR') ? '' : 'static-text-disabled';
            var typeOfCare = this.model.getCareId();

            this.model.set('facility', model);

            return new SingleItemView({
                className: className,
                model: model,
                fDNS   el: this.model,
                expressMessage: this.getExpressMessage(typeOfCare, model),
            });
        },

        /**
         * Creates the section view
         * @return {Marionette}
         */
        getSectionView: function() {
            var clinicsInstitutionCodes = this.model.getCareId() === PRIMARY_CARE_ID ?
                this.getClinicsInstitutionCodes() :
                [];
            
            return new SectionView({
                model: this.model,
                patientEnrolledFacilities: this.collection,
                clinicsInstitutionCodes: clinicsInstitutionCodes,
                googleAnalyticsProvider: this.googleAnalyticsProvider,
            });
        },

        /**
         * Collates all clinics amongst all facilities
         * @return {void}
         */
        collateClinicsForFacilities: function() {
            var typeOfCare = this.model.getCareId();
            var cityState = this.model.get('cityState');
            var parentCode = cityState.get('institutionCode');
            var view;
            
            var institutionCodes = this.collection.map(function(model) {
                return model.get('institutionCode');
            });

            var clinicsPromise = this.clinicsCollection.fetchAllClinics(parentCode, institutionCodes, typeOfCare);
            var region = this.getRegion('section');
            clinicsPromise.then(function(results) {
                this.clinicsCollection.reset(results);

                view = institutionCodes.length === 1 ? this.getSingleItemView() : this.getSectionView();
                region.show(view);

                view.injectPacTeam();
            }.bind(this));

            this.model.set('unfilteredClinics', this.clinicsCollection);
        },

        /**
         * Gets all the institution codes for the unfiltered clinics
         * @return {array} clinicsInstitutionCodes a list of all the institution
         * codes for the clinics
         */
        getClinicsInstitutionCodes: function() {
            var clinics = this.model.get('unfilteredClinics') || new ClinicsCollection();
            var clinicsInstitutionCodes = clinics.map(function(clinic) {
                return clinic.get('institutionCode');
            });

            return clinicsInstitutionCodes;
        },
    });

    View.TITLE = 'Location';
    View.STEP_NUMBER = 3;

    return View;
});
T